﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/PersonalizeRequest.h>
#include <aws/personalize/Personalize_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Personalize {
namespace Model {

/**
 */
class ListCampaignsRequest : public PersonalizeRequest {
 public:
  AWS_PERSONALIZE_API ListCampaignsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListCampaigns"; }

  AWS_PERSONALIZE_API Aws::String SerializePayload() const override;

  AWS_PERSONALIZE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the solution to list the campaigns for.
   * When a solution is not specified, all the campaigns associated with the account
   * are listed.</p>
   */
  inline const Aws::String& GetSolutionArn() const { return m_solutionArn; }
  inline bool SolutionArnHasBeenSet() const { return m_solutionArnHasBeenSet; }
  template <typename SolutionArnT = Aws::String>
  void SetSolutionArn(SolutionArnT&& value) {
    m_solutionArnHasBeenSet = true;
    m_solutionArn = std::forward<SolutionArnT>(value);
  }
  template <typename SolutionArnT = Aws::String>
  ListCampaignsRequest& WithSolutionArn(SolutionArnT&& value) {
    SetSolutionArn(std::forward<SolutionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token returned from the previous call to <a
   * href="https://docs.aws.amazon.com/personalize/latest/dg/API_ListCampaigns.html">ListCampaigns</a>
   * for getting the next set of campaigns (if they exist).</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListCampaignsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of campaigns to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListCampaignsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_solutionArn;
  bool m_solutionArnHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
